<?php
/**
 * PWA Navigation Template
 * 
 * Wird in alle Admin-Seiten im PWA-Modus eingefügt.
 * Enthält: Top Header, Hamburger Menu, Bottom Navigation
 *
 * @package SimpleBookingCalendar
 * @since 1.2.0
 */

// Sicherheitsmaßnahme: Direkten Zugriff verhindern
if (!defined('ABSPATH')) {
    exit;
}

// Nur im PWA-Modus anzeigen
if (!simpbook_should_show_pwa_nav()) {
    return;
}

$simpbook_current_page = simpbook_get_current_page();
$current_user = wp_get_current_user();
$simpbook_mitarbeiter = simpbook_get_current_mitarbeiter();
$simpbook_page_title = simpbook_get_pwa_page_title();
$simpbook_show_back = simpbook_show_back_button();

// PWA Settings laden - nutze die korrekten Option-Namen
$simpbook_app_name = get_option('simpbook_firmen_name', get_bloginfo('name'));
$simpbook_theme_color = get_option('simpbook_pwa_theme_color', '#667eea');

// Einen dunkleren Farbton für den Gradient berechnen
if (!function_exists('simpbook_darken_color')) {
    function simpbook_darken_color($hex, $percent = 20) {
        $hex = ltrim($hex, '#');
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
        
        $r = max(0, $r - ($r * $percent / 100));
        $g = max(0, $g - ($g * $percent / 100));
        $b = max(0, $b - ($b * $percent / 100));
        
        return sprintf('#%02x%02x%02x', $r, $g, $b);
    }
}
$simpbook_theme_color_dark = simpbook_darken_color($simpbook_theme_color, 25);
?>

<!-- PWA Inline Script - Sofort verfügbare Fallback-Funktionen für onclick Handler -->
<script>
(function() {
    console.log('PWA: Inline Script wird ausgeführt');
    
    // Sofort ein minimales simpbookPWA Objekt erstellen
    window.simpbookPWA = {
        menuOpen: false,
        
        toggleMenu: function() {
            console.log('PWA: toggleMenu() aufgerufen, current state:', this.menuOpen);
            var menu = document.getElementById('simpbook-pwa-menu');
            var overlay = document.getElementById('simpbook-pwa-overlay');
            console.log('PWA: Menu Element:', menu, 'Overlay Element:', overlay);
            
            if (menu && overlay) {
                this.menuOpen = !this.menuOpen;
                if (this.menuOpen) {
                    menu.classList.add('open');
                    overlay.classList.add('open');
                    document.body.style.overflow = 'hidden';
                    console.log('PWA: Menu GEÖFFNET');
                } else {
                    menu.classList.remove('open');
                    overlay.classList.remove('open');
                    document.body.style.overflow = '';
                    console.log('PWA: Menu GESCHLOSSEN');
                }
            } else {
                console.error('PWA: Menu oder Overlay nicht gefunden!');
            }
        },
        
        closeMenu: function() {
            console.log('PWA: closeMenu() aufgerufen');
            var menu = document.getElementById('simpbook-pwa-menu');
            var overlay = document.getElementById('simpbook-pwa-overlay');
            
            if (menu && overlay) {
                this.menuOpen = false;
                menu.classList.remove('open');
                overlay.classList.remove('open');
                document.body.style.overflow = '';
                console.log('PWA: Menu geschlossen');
            }
        },
        
        goBack: function() {
            console.log('PWA: goBack() aufgerufen');
            if (window.history.length > 1) {
                window.history.back();
            } else {
                window.location.href = '<?php echo esc_js(simpbook_pwa_url('simpbook-dashboard')); ?>';
            }
        },
        
        handleLogout: function() {
            console.log('PWA: handleLogout() aufgerufen');
            if (confirm('<?php echo esc_js(__('Möchten Sie sich wirklich abmelden?', 'simple-appointment-booking')); ?>')) {
                window.location.href = '<?php echo esc_js(wp_logout_url(home_url('/simple-booking-login/?pwa=1&logged_out=1'))); ?>';
            }
        }
    };
    
    console.log('PWA: simpbookPWA Objekt erstellt:', window.simpbookPWA);
})();
</script>

<!-- PWA Top Header -->
<header class="simpbook-pwa-header" style="background: linear-gradient(135deg, <?php echo esc_attr($simpbook_theme_color); ?> 0%, <?php echo esc_attr($simpbook_theme_color_dark); ?> 100%);">
    <button type="button" class="simpbook-pwa-back-btn" id="simpbook-pwa-back-btn" <?php echo $simpbook_show_back ? '' : 'style="visibility:hidden;"'; ?>>
        <span class="dashicons dashicons-arrow-left-alt2"></span>
    </button>
    
    <h1 class="simpbook-pwa-title"><?php echo esc_html($simpbook_page_title); ?></h1>
    
    <button type="button" class="simpbook-pwa-menu-btn" id="simpbook-pwa-menu-btn">
        <span class="dashicons dashicons-menu"></span>
    </button>
</header>

<!-- PWA Event Listeners - Direkt nach Header -->
<script>
(function() {
    console.log('PWA: Event Listener Script wird ausgeführt');
    
    function attachListeners() {
        console.log('PWA: attachListeners() wird ausgeführt');
        
        // Menu Button
        var menuBtn = document.getElementById('simpbook-pwa-menu-btn');
        console.log('PWA: Menu Button gefunden:', menuBtn);
        
        if (menuBtn) {
            // Alle alten Listener entfernen durch Klonen
            var newMenuBtn = menuBtn.cloneNode(true);
            menuBtn.parentNode.replaceChild(newMenuBtn, menuBtn);
            
            newMenuBtn.addEventListener('click', function(e) {
                console.log('PWA: Menu Button CLICK EVENT');
                e.preventDefault();
                e.stopPropagation();
                if (window.simpbookPWA && typeof window.simpbookPWA.toggleMenu === 'function') {
                    window.simpbookPWA.toggleMenu();
                } else {
                    console.error('PWA: simpbookPWA.toggleMenu nicht verfügbar!');
                }
            }, true); // useCapture = true
            
            console.log('PWA: Menu Button Listener hinzugefügt');
        }
        
        // Back Button
        var backBtn = document.getElementById('simpbook-pwa-back-btn');
        if (backBtn) {
            var newBackBtn = backBtn.cloneNode(true);
            backBtn.parentNode.replaceChild(newBackBtn, backBtn);
            
            newBackBtn.addEventListener('click', function(e) {
                console.log('PWA: Back Button CLICK EVENT');
                e.preventDefault();
                if (window.simpbookPWA && typeof window.simpbookPWA.goBack === 'function') {
                    window.simpbookPWA.goBack();
                }
            }, true);
            
            console.log('PWA: Back Button Listener hinzugefügt');
        }
        
        // Overlay
        var overlay = document.getElementById('simpbook-pwa-overlay');
        console.log('PWA: Overlay gefunden:', overlay);
        
        if (overlay) {
            var newOverlay = overlay.cloneNode(true);
            overlay.parentNode.replaceChild(newOverlay, overlay);
            
            newOverlay.addEventListener('click', function(e) {
                console.log('PWA: Overlay CLICK EVENT');
                if (window.simpbookPWA && typeof window.simpbookPWA.closeMenu === 'function') {
                    window.simpbookPWA.closeMenu();
                }
            }, true);
            
            console.log('PWA: Overlay Listener hinzugefügt');
        }
        
        // Logout Link (kommt später im DOM, muss nach Menu existieren)
        setTimeout(function() {
            var logoutLink = document.getElementById('simpbook-pwa-logout-link');
            if (logoutLink) {
                var newLogoutLink = logoutLink.cloneNode(true);
                logoutLink.parentNode.replaceChild(newLogoutLink, logoutLink);
                
                newLogoutLink.addEventListener('click', function(e) {
                    console.log('PWA: Logout CLICK EVENT');
                    e.preventDefault();
                    if (window.simpbookPWA && typeof window.simpbookPWA.handleLogout === 'function') {
                        window.simpbookPWA.handleLogout();
                    }
                }, true);
                
                console.log('PWA: Logout Link Listener hinzugefügt');
            }
        }, 100);
    }
    
    // Sofort versuchen
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', attachListeners);
        console.log('PWA: Wartet auf DOMContentLoaded');
    } else {
        console.log('PWA: DOM bereits geladen, führe attachListeners() aus');
        attachListeners();
    }
    
    // Zusätzlicher Fallback
    setTimeout(attachListeners, 200);
})();
</script>

<!-- Hamburger Menu Overlay -->
<div class="simpbook-pwa-overlay" id="simpbook-pwa-overlay"></div>

<!-- Hamburger Menu -->
<aside class="simpbook-pwa-menu" id="simpbook-pwa-menu">
    <div class="simpbook-pwa-menu-header" style="background: linear-gradient(135deg, <?php echo esc_attr($simpbook_theme_color); ?> 0%, <?php echo esc_attr($simpbook_theme_color_dark); ?> 100%);">
        <div class="simpbook-pwa-user-info">
            <?php if ($simpbook_mitarbeiter && !empty($simpbook_mitarbeiter->profilfoto_url)) : ?>
                <img src="<?php echo esc_url($simpbook_mitarbeiter->profilfoto_url); ?>" 
                     alt="<?php echo esc_attr($current_user->display_name); ?>"
                     class="simpbook-pwa-avatar">
            <?php else : ?>
                <div class="simpbook-pwa-avatar-placeholder">
                    <?php echo esc_html(mb_substr($current_user->display_name, 0, 1)); ?>
                </div>
            <?php endif; ?>
            <div class="simpbook-pwa-user-details">
                <strong><?php echo esc_html($current_user->display_name); ?></strong>
                <small><?php echo esc_html($current_user->user_email); ?></small>
            </div>
        </div>
    </div>
    
    <nav class="simpbook-pwa-menu-nav">
        <a href="<?php echo esc_url(simpbook_pwa_url('simpbook-dashboard')); ?>" 
           class="<?php echo $simpbook_current_page === 'simpbook-dashboard' ? 'active' : ''; ?>">
            <span class="dashicons dashicons-dashboard"></span>
            <?php esc_html_e('Dashboard', 'simple-appointment-booking'); ?>
        </a>
        
        <a href="<?php echo esc_url(simpbook_pwa_url('simpbook-reservierungen-neu')); ?>">
            <span class="dashicons dashicons-plus-alt"></span>
            <?php esc_html_e('Neue Buchung', 'simple-appointment-booking'); ?>
        </a>
        
        <a href="<?php echo esc_url(simpbook_pwa_url('simpbook-dienstleistungen')); ?>"
           class="<?php echo $simpbook_current_page === 'simpbook-dienstleistungen' ? 'active' : ''; ?>">
            <span class="dashicons dashicons-admin-tools"></span>
            <?php esc_html_e('Dienstleistungen', 'simple-appointment-booking'); ?>
        </a>
        
        <a href="<?php echo esc_url(simpbook_pwa_url('simpbook-mitarbeiter')); ?>"
           class="<?php echo $simpbook_current_page === 'simpbook-mitarbeiter' ? 'active' : ''; ?>">
            <span class="dashicons dashicons-groups"></span>
            <?php esc_html_e('Mitarbeiter', 'simple-appointment-booking'); ?>
        </a>
        
        <a href="<?php echo esc_url(simpbook_pwa_url('simpbook-statistiken')); ?>"
           class="<?php echo $simpbook_current_page === 'simpbook-statistiken' ? 'active' : ''; ?>">
            <span class="dashicons dashicons-chart-bar"></span>
            <?php esc_html_e('Statistiken', 'simple-appointment-booking'); ?>
        </a>
        
        <?php if (current_user_can('manage_options')) : ?>
        <hr>
        <a href="<?php echo esc_url(simpbook_pwa_url('simpbook-reservierungen-einstellungen')); ?>"
           class="<?php echo $simpbook_current_page === 'simpbook-reservierungen-einstellungen' ? 'active' : ''; ?>">
            <span class="dashicons dashicons-admin-settings"></span>
            <?php esc_html_e('Einstellungen', 'simple-appointment-booking'); ?>
        </a>
        <?php endif; ?>
        
        <!-- Push Notification Toggle (wird via JS gesteuert) -->
        <div id="simpbook-pwa-push-nav-item" style="display: none;">
            <hr>
            <a href="#" class="simpbook-pwa-nav-item" id="simpbook-pwa-enable-push-btn">
                <span class="dashicons dashicons-bell"></span>
                <?php esc_html_e('Push-Nachrichten aktivieren', 'simple-appointment-booking'); ?>
            </a>
        </div>
        
        <hr>
        
        <a href="#" id="simpbook-pwa-logout-link" class="simpbook-pwa-logout">
            <span class="dashicons dashicons-exit"></span>
            <?php esc_html_e('Abmelden', 'simple-appointment-booking'); ?>
        </a>
    </nav>
    
    <div class="simpbook-pwa-menu-footer">
        <small><?php echo esc_html($simpbook_app_name); ?> &bull; v<?php echo esc_html(SIMPBOOK_VERSION); ?></small>
    </div>
</aside>

<!-- PWA Bottom Navigation -->
<nav class="simpbook-pwa-bottom-nav">
    <a href="<?php echo esc_url(simpbook_pwa_url('simpbook-dashboard')); ?>" 
       class="simpbook-pwa-nav-item <?php echo $simpbook_current_page === 'simpbook-dashboard' ? 'active' : ''; ?>">
        <span class="simpbook-pwa-nav-icon dashicons dashicons-admin-home"></span>
        <span class="simpbook-pwa-nav-label"><?php esc_html_e('Home', 'simple-appointment-booking'); ?></span>
    </a>
    
    <a href="<?php echo esc_url(simpbook_pwa_url('simpbook-kalender')); ?>" 
       class="simpbook-pwa-nav-item <?php echo $simpbook_current_page === 'simpbook-kalender' ? 'active' : ''; ?>">
        <span class="simpbook-pwa-nav-icon dashicons dashicons-calendar-alt"></span>
        <span class="simpbook-pwa-nav-label"><?php esc_html_e('Kalender', 'simple-appointment-booking'); ?></span>
    </a>
    
    <a href="<?php echo esc_url(simpbook_pwa_url('simpbook-reservierungen')); ?>" 
       class="simpbook-pwa-nav-item <?php echo $simpbook_current_page === 'simpbook-reservierungen' ? 'active' : ''; ?>">
        <span class="simpbook-pwa-nav-icon dashicons dashicons-list-view"></span>
        <span class="simpbook-pwa-nav-label"><?php esc_html_e('Buchungen', 'simple-appointment-booking'); ?></span>
    </a>
    
    <?php if (!$simpbook_mitarbeiter) : // "Mehr" nur für Admins anzeigen, nicht für Mitarbeiter ?>
    <a href="<?php echo esc_url(simpbook_pwa_url('simpbook-reservierungen-einstellungen')); ?>" 
       class="simpbook-pwa-nav-item <?php echo $simpbook_current_page === 'simpbook-reservierungen-einstellungen' ? 'active' : ''; ?>">
        <span class="simpbook-pwa-nav-icon dashicons dashicons-admin-generic"></span>
        <span class="simpbook-pwa-nav-label"><?php esc_html_e('Mehr', 'simple-appointment-booking'); ?></span>
    </a>
    <?php endif; ?>
</nav>

<?php
// Body-Class für PWA-Modus hinzufügen
add_filter('admin_body_class', function($classes) {
    return $classes . ' simpbook-pwa-mode';
});
?>
