<?php
/**
 * PWA Login Template - PocketBooking Pro
 * 
 * Gebrandetes Login-Formular für die PWA.
 * Kein WordPress-Feeling - reine App-Optik.
 *
 * @package SimpleBookingCalendar
 * @since 1.2.0
 */

if (!defined('ABSPATH')) {
    exit;
}

// Theme-Farbe aus Einstellungen oder Default
$simpbook_theme_color = get_option('simpbook_pwa_theme_color', '#667eea');
$simpbook_business_name = get_option('simpbook_business_name', get_bloginfo('name'));
$simpbook_custom_logo_id = get_theme_mod('custom_logo');
$simpbook_logo_url = $simpbook_custom_logo_id ? wp_get_attachment_image_url($simpbook_custom_logo_id, 'medium') : '';

// Error aus Auth-Class
$simpbook_error_message = isset($error) ? $error : '';

?>
<!DOCTYPE html>
<html <?php language_attributes(); ?>>
<head>
    <meta charset="<?php bloginfo('charset'); ?>">
    <meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no">
    <meta name="theme-color" content="<?php echo esc_attr($simpbook_theme_color); ?>">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="black-translucent">
    <title><?php echo esc_html($simpbook_business_name); ?> - <?php esc_html_e('Login', 'simple-appointment-booking'); ?></title>
    
    <!-- PWA Manifest -->
    <link rel="manifest" href="<?php echo esc_url(admin_url('admin-ajax.php?action=simpbook_pwa_manifest')); ?>">
    
    <style>
        * {
            box-sizing: border-box;
            margin: 0;
            padding: 0;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, Cantarell, sans-serif;
            background: linear-gradient(135deg, <?php echo esc_attr($simpbook_theme_color); ?> 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .login-container {
            width: 100%;
            max-width: 400px;
            background: white;
            border-radius: 20px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            overflow: hidden;
        }
        
        .login-header {
            background: linear-gradient(135deg, <?php echo esc_attr($simpbook_theme_color); ?> 0%, #764ba2 100%);
            padding: 40px 30px;
            text-align: center;
            color: white;
        }
        
        .login-logo {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            background: rgba(255, 255, 255, 0.2);
            margin: 0 auto 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            backdrop-filter: blur(10px);
        }
        
        .login-logo img {
            max-width: 60px;
            max-height: 60px;
            border-radius: 50%;
        }
        
        .login-logo svg {
            width: 40px;
            height: 40px;
            fill: white;
        }
        
        .login-title {
            font-size: 24px;
            font-weight: 600;
            margin-bottom: 5px;
        }
        
        .login-subtitle {
            font-size: 14px;
            opacity: 0.9;
        }
        
        .login-form {
            padding: 30px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            font-size: 14px;
            font-weight: 500;
            color: #333;
            margin-bottom: 8px;
        }
        
        .form-group input {
            width: 100%;
            padding: 14px 16px;
            font-size: 16px;
            border: 2px solid #e0e0e0;
            border-radius: 10px;
            transition: all 0.3s ease;
            outline: none;
        }
        
        .form-group input:focus {
            border-color: <?php echo esc_attr($simpbook_theme_color); ?>;
            box-shadow: 0 0 0 3px <?php echo esc_attr($simpbook_theme_color); ?>33;
        }
        
        .form-group input::placeholder {
            color: #999;
        }
        
        .form-group-checkbox {
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .form-group-checkbox input[type="checkbox"] {
            width: 20px;
            height: 20px;
            accent-color: <?php echo esc_attr($simpbook_theme_color); ?>;
        }
        
        .form-group-checkbox label {
            margin-bottom: 0;
            cursor: pointer;
        }
        
        .login-btn {
            width: 100%;
            padding: 16px;
            font-size: 16px;
            font-weight: 600;
            color: white;
            background: linear-gradient(135deg, <?php echo esc_attr($simpbook_theme_color); ?> 0%, #764ba2 100%);
            border: none;
            border-radius: 10px;
            cursor: pointer;
            transition: all 0.3s ease;
            margin-top: 10px;
        }
        
        .login-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 30px <?php echo esc_attr($simpbook_theme_color); ?>66;
        }
        
        .login-btn:active {
            transform: translateY(0);
        }
        
        .login-btn:disabled {
            opacity: 0.7;
            cursor: not-allowed;
            transform: none;
        }
        
        .error-message {
            background: #ffebee;
            color: #c62828;
            padding: 12px 16px;
            border-radius: 10px;
            margin-bottom: 20px;
            font-size: 14px;
            border-left: 4px solid #c62828;
        }
        
        .success-message {
            background: #e8f5e9;
            color: #2e7d32;
            padding: 12px 16px;
            border-radius: 10px;
            margin-bottom: 20px;
            font-size: 14px;
            border-left: 4px solid #2e7d32;
        }
        
        .login-footer {
            text-align: center;
            padding: 20px 30px 30px;
            border-top: 1px solid #f0f0f0;
        }
        
        .login-footer a {
            color: <?php echo esc_attr($simpbook_theme_color); ?>;
            text-decoration: none;
            font-size: 14px;
        }
        
        .login-footer a:hover {
            text-decoration: underline;
        }
        
        .password-toggle {
            position: relative;
        }
        
        .password-toggle input {
            padding-right: 50px;
        }
        
        .password-toggle-btn {
            position: absolute;
            right: 12px;
            top: 50%;
            transform: translateY(-50%);
            background: none;
            border: none;
            color: #666;
            cursor: pointer;
            padding: 8px;
        }
        
        .loading {
            display: none;
            text-align: center;
            padding: 20px;
        }
        
        .loading.active {
            display: block;
        }
        
        .spinner {
            width: 40px;
            height: 40px;
            border: 3px solid #f0f0f0;
            border-top-color: <?php echo esc_attr($simpbook_theme_color); ?>;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin: 0 auto 10px;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        /* Responsive */
        @media (max-width: 440px) {
            .login-container {
                border-radius: 0;
                min-height: 100vh;
            }
            
            body {
                padding: 0;
            }
        }
        
        /* Safe Area für Notch */
        @supports (padding: env(safe-area-inset-top)) {
            .login-header {
                padding-top: calc(40px + env(safe-area-inset-top));
            }
        }
    </style>
</head>
<body>
    <div class="login-container">
        <div class="login-header">
            <div class="login-logo">
                <?php if ($simpbook_logo_url): ?>
                    <img src="<?php echo esc_url($simpbook_logo_url); ?>" alt="<?php echo esc_attr($simpbook_business_name); ?>">
                <?php else: ?>
                    <svg viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                        <path d="M19 3h-1V1h-2v2H8V1H6v2H5c-1.11 0-1.99.9-1.99 2L3 19c0 1.1.89 2 2 2h14c1.1 0 2-.9 2-2V5c0-1.1-.9-2-2-2zm0 16H5V8h14v11zM9 10H7v2h2v-2zm4 0h-2v2h2v-2zm4 0h-2v2h2v-2zm-8 4H7v2h2v-2zm4 0h-2v2h2v-2zm4 0h-2v2h2v-2z"/>
                    </svg>
                <?php endif; ?>
            </div>
            <h1 class="login-title"><?php echo esc_html($simpbook_business_name); ?></h1>
            <p class="login-subtitle"><?php esc_html_e('Buchungs-App', 'simple-appointment-booking'); ?></p>
        </div>
        
        <form class="login-form" id="pwa-login-form" method="post" action="">
            <?php wp_nonce_field('simpbook_pwa_login', 'simpbook_pwa_login_nonce'); ?>
            
            <?php if (!empty($simpbook_error_message)): ?>
                <div class="error-message">
                    <?php echo esc_html($simpbook_error_message); ?>
                </div>
            <?php endif; ?>
            
            <div id="ajax-messages"></div>
            
            <div class="form-group">
                <label for="username"><?php esc_html_e('Benutzername oder E-Mail', 'simple-appointment-booking'); ?></label>
                <input 
                    type="text" 
                    id="username" 
                    name="username" 
                    autocomplete="username"
                    autocapitalize="none"
                    required
                    placeholder="<?php esc_attr_e('Benutzername eingeben', 'simple-appointment-booking'); ?>"
                >
            </div>
            
            <div class="form-group">
                <label for="password"><?php esc_html_e('Passwort', 'simple-appointment-booking'); ?></label>
                <div class="password-toggle">
                    <input 
                        type="password" 
                        id="password" 
                        name="password" 
                        autocomplete="current-password"
                        required
                        placeholder="<?php esc_attr_e('Passwort eingeben', 'simple-appointment-booking'); ?>"
                    >
                    <button type="button" class="password-toggle-btn" onclick="togglePassword()">
                        <svg id="eye-icon" width="20" height="20" viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12 4.5C7 4.5 2.73 7.61 1 12c1.73 4.39 6 7.5 11 7.5s9.27-3.11 11-7.5c-1.73-4.39-6-7.5-11-7.5zM12 17c-2.76 0-5-2.24-5-5s2.24-5 5-5 5 2.24 5 5-2.24 5-5 5zm0-8c-1.66 0-3 1.34-3 3s1.34 3 3 3 3-1.34 3-3-1.34-3-3-3z"/>
                        </svg>
                    </button>
                </div>
            </div>
            
            <div class="form-group form-group-checkbox">
                <input type="checkbox" id="remember" name="remember" value="1">
                <label for="remember"><?php esc_html_e('Angemeldet bleiben', 'simple-appointment-booking'); ?></label>
            </div>
            
            <button type="submit" class="login-btn" id="login-btn">
                <?php esc_html_e('Anmelden', 'simple-appointment-booking'); ?>
            </button>
            
            <div class="loading" id="loading">
                <div class="spinner"></div>
                <p><?php esc_html_e('Anmeldung läuft...', 'simple-appointment-booking'); ?></p>
            </div>
        </form>
        
        <div class="login-footer">
            <a href="<?php echo esc_url(wp_lostpassword_url()); ?>">
                <?php esc_html_e('Passwort vergessen?', 'simple-appointment-booking'); ?>
            </a>
        </div>
    </div>
    
    <script>
        // Password Toggle
        function togglePassword() {
            const input = document.getElementById('password');
            const icon = document.getElementById('eye-icon');
            
            if (input.type === 'password') {
                input.type = 'text';
                icon.innerHTML = '<path d="M12 7c2.76 0 5 2.24 5 5 0 .65-.13 1.26-.36 1.83l2.92 2.92c1.51-1.26 2.7-2.89 3.43-4.75-1.73-4.39-6-7.5-11-7.5-1.4 0-2.74.25-3.98.7l2.16 2.16C10.74 7.13 11.35 7 12 7zM2 4.27l2.28 2.28.46.46C3.08 8.3 1.78 10.02 1 12c1.73 4.39 6 7.5 11 7.5 1.55 0 3.03-.3 4.38-.84l.42.42L19.73 22 21 20.73 3.27 3 2 4.27zM7.53 9.8l1.55 1.55c-.05.21-.08.43-.08.65 0 1.66 1.34 3 3 3 .22 0 .44-.03.65-.08l1.55 1.55c-.67.33-1.41.53-2.2.53-2.76 0-5-2.24-5-5 0-.79.2-1.53.53-2.2zm4.31-.78l3.15 3.15.02-.16c0-1.66-1.34-3-3-3l-.17.01z"/>';
            } else {
                input.type = 'password';
                icon.innerHTML = '<path d="M12 4.5C7 4.5 2.73 7.61 1 12c1.73 4.39 6 7.5 11 7.5s9.27-3.11 11-7.5c-1.73-4.39-6-7.5-11-7.5zM12 17c-2.76 0-5-2.24-5-5s2.24-5 5-5 5 2.24 5 5-2.24 5-5 5zm0-8c-1.66 0-3 1.34-3 3s1.34 3 3 3 3-1.34 3-3-1.34-3-3-3z"/>';
            }
        }
        
        // Optional: AJAX Login für bessere UX
        document.getElementById('pwa-login-form').addEventListener('submit', function(e) {
            // Form wird normal submitted (server-side)
            const btn = document.getElementById('login-btn');
            const loading = document.getElementById('loading');
            
            btn.disabled = true;
            btn.style.display = 'none';
            loading.classList.add('active');
        });
        
        // Auto-focus Username
        document.getElementById('username').focus();
        
        // Service Worker registrieren (für PWA)
        if ('serviceWorker' in navigator) {
            navigator.serviceWorker.register('<?php echo esc_js(SIMPBOOK_PLUGIN_URL . 'service-worker.js'); ?>')
                .then(function(registration) {
                    console.log('SW registered:', registration.scope);
                })
                .catch(function(error) {
                    console.log('SW registration failed:', error);
                });
        }
    </script>
</body>
</html>
