/**
 * js/gruppierungen.js
 * JavaScript für die Gruppierungen/Kategorien Admin-Seite
 */

(function($) {
    'use strict';

    // DOM Ready
    $(document).ready(function() {
        initModal();
        initRowClick();
        initEditButtons();
        initDeleteButtons();
        initColorPicker();
    });

    /**
     * Color Picker initialisieren
     */
    function initColorPicker() {
        const $colorPicker = $('#kategorie_color_picker');
        const $textInput = $('#simpbook_kategorie_farbe');
        const $applyBtn = $('#apply_kategorie_color_btn');

        // Synchronisiere Color Picker -> Text Input beim Ändern
        $colorPicker.on('input', function() {
            $textInput.val($(this).val());
        });

        // Synchronisiere Text Input -> Color Picker beim Tippen
        $textInput.on('input', function() {
            const value = $(this).val();
            if (/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/.test(value)) {
                $colorPicker.val(value);
            }
        });

        // Apply Button
        $applyBtn.on('click', function() {
            const value = $textInput.val();
            if (/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/.test(value)) {
                $colorPicker.val(value);
            } else {
                alert('Bitte geben Sie eine gültige Hex-Farbe ein (z.B. #FFFFFF)');
            }
        });
    }

    /**
     * Modal initialisieren
     */
    function initModal() {
        const $modal = $('#kategorie-modal');
        const $addBtn = $('#add-new-kategorie-btn');
        const $closeBtn = $modal.find('.simpbook-modal-close, .simpbook-modal-cancel');
        const $overlay = $modal.find('.simpbook-modal-overlay');

        // Öffnen bei Klick auf "Neue Gruppierung"
        $addBtn.on('click', function() {
            openModal();
        });

        // Schließen
        $closeBtn.on('click', function() {
            closeModal();
        });

        $overlay.on('click', function() {
            closeModal();
        });

        // ESC-Taste
        $(document).on('keydown', function(e) {
            if (e.key === 'Escape' && $modal.is(':visible')) {
                closeModal();
            }
        });
    }

    /**
     * Modal öffnen
     */
    function openModal(kategorieData) {
        const $modal = $('#kategorie-modal');
        const $title = $('#kategorie-modal-title');
        const $form = $('#kategorie-form');

        // Form zurücksetzen
        $form[0].reset();
        $('#simpbook_kategorie_id').val('');

        if (kategorieData) {
            // Bearbeitungsmodus
            $title.text(simpbookGruppierungenL10n.editTitle || 'Gruppierung bearbeiten');
            $('#simpbook_kategorie_id').val(kategorieData.id);
            $('#simpbook_kategorie_name').val(kategorieData.name);
            $('#simpbook_kategorie_beschreibung').val(kategorieData.beschreibung);
            $('#simpbook_kategorie_sortierung').val(kategorieData.sortierung);
            
            // Farbe setzen (neuer Color Picker)
            const color = kategorieData.farbe || '#2271b1';
            $('#simpbook_kategorie_farbe').val(color);
            $('#kategorie_color_picker').val(color);
        } else {
            // Neuer Eintrag
            $title.text(simpbookGruppierungenL10n.newTitle || 'Neue Gruppierung');
            // Standardfarbe setzen
            $('#simpbook_kategorie_farbe').val('#2271b1');
            $('#kategorie_color_picker').val('#2271b1');
        }

        $modal.fadeIn(200);
        $('#simpbook_kategorie_name').focus();
    }

    /**
     * Modal schließen
     */
    function closeModal() {
        $('#kategorie-modal').fadeOut(200);
    }

    /**
     * Zeilen-Klick für Bearbeitung (Desktop)
     */
    function initRowClick() {
        $('.kategorie-row-clickable').on('click', function(e) {
            // Nicht bei Klick auf Buttons
            if ($(e.target).closest('button, a').length) {
                return;
            }

            const kategorieData = $(this).data('kategorie-data');
            if (kategorieData) {
                openModal(kategorieData);
            }
        });

        // Mobile Karten
        $('.gruppierung-karte').on('click', function(e) {
            // Nicht bei Klick auf Buttons
            if ($(e.target).closest('button, a').length) {
                return;
            }

            const kategorieData = $(this).data('kategorie-data');
            if (kategorieData) {
                openModal(kategorieData);
            }
        });
    }

    /**
     * Edit-Buttons initialisieren
     */
    function initEditButtons() {
        $('.edit-kategorie-btn').on('click', function(e) {
            e.preventDefault();
            e.stopPropagation();

            const $row = $(this).closest('[data-kategorie-data]');
            const kategorieData = $row.data('kategorie-data');

            if (kategorieData) {
                openModal(kategorieData);
            }
        });
    }

    /**
     * Delete-Buttons mit Bestätigungsabfrage initialisieren
     */
    function initDeleteButtons() {
        $('.delete-kategorie-btn').on('click', function(e) {
            e.preventDefault();
            e.stopPropagation();

            const $btn = $(this);
            const deleteUrl = $btn.attr('href');
            const $row = $btn.closest('[data-kategorie-id]');
            
            const kategorieName = $row.data('kategorie-name') || 'diese Gruppierung';
            const dlCount = parseInt($row.data('dl-count')) || 0;
            const dlNames = $row.data('dl-names') || [];

            // Nachricht erstellen
            let message = '';
            
            if (dlCount > 0) {
                const confirmQuestion = simpbookGruppierungenL10n.confirmDeleteQuestion || 'Sind Sie sicher, dass Sie die Gruppierung "{name}" löschen möchten?';
                message = '<strong>' + confirmQuestion.replace('{name}', escapeHtml(kategorieName)) + '</strong><br><br>';
                
                const servicesConnected = dlCount === 1 
                    ? (simpbookGruppierungenL10n.oneServiceConnected || 'Es ist noch <strong>1 Dienstleistung</strong> mit dieser Kategorie verbunden:')
                    : (simpbookGruppierungenL10n.multipleServicesConnected || 'Es sind noch <strong>{count} Dienstleistungen</strong> mit dieser Kategorie verbunden:').replace('{count}', dlCount);
                    
                message += '<span style="color: #ffcccc;">⚠️ ' + servicesConnected + '</span>';
                message += '<ul style="text-align: left; margin: 15px auto; max-width: 300px; padding-left: 20px;">';
                dlNames.forEach(function(name) {
                    message += '<li style="margin: 5px 0;">• ' + escapeHtml(name) + '</li>';
                });
                message += '</ul>';
                message += '<span style="font-size: 13px; opacity: 0.9;">' + (simpbookGruppierungenL10n.servicesNotDeleted || 'Die Dienstleistungen werden nicht gelöscht, nur die Kategorie-Zuordnung wird entfernt.') + '</span>';
            } else {
                const confirmDeleteSimple = simpbookGruppierungenL10n.confirmDeleteSimple || 'Möchten Sie die Gruppierung "{name}" wirklich löschen?';
                message = confirmDeleteSimple.replace('{name}', '<strong>' + escapeHtml(kategorieName) + '</strong>');
            }

            // Custom Confirm Dialog verwenden
            if (typeof window.simpbookConfirm === 'function') {
                window.simpbookConfirm(message, simpbookGruppierungenL10n.deleteTitle || 'Gruppierung löschen').then(function(confirmed) {
                    if (confirmed) {
                        window.location.href = deleteUrl;
                    }
                });
            } else {
                // Fallback auf Standard-Confirm
                if (confirm('Möchten Sie die Gruppierung "' + kategorieName + '" wirklich löschen? ' + (dlCount > 0 ? dlCount + ' Dienstleistungen sind noch zugeordnet.' : ''))) {
                    window.location.href = deleteUrl;
                }
            }
        });
    }

    /**
     * HTML-Entities escapen
     */
    function escapeHtml(text) {
        const map = {
            '&': '&amp;',
            '<': '&lt;',
            '>': '&gt;',
            '"': '&quot;',
            "'": '&#039;'
        };
        return text.replace(/[&<>"']/g, function(m) { return map[m]; });
    }

})(jQuery);
