/**
 * Admin JavaScript for Custom Confirm/Alert Dialogs
 * 
 * @package Simple_Booking_Calendar
 */

// Globale Konfiguration für Dialog-Buttons (wird von WordPress überschrieben)
window.simpbookDialogConfig = window.simpbookDialogConfig || {
	buttonCancel: 'Abbrechen',
	buttonConfirm: 'Bestätigen',
	buttonOk: 'OK'
};

(function($) {
	'use strict';

	// Custom Modal Dialog System
	// Ersetzt Browser confirm() und alert() mit benutzerdefinierten Modals
	
	/**
	 * Zeigt einen benutzerdefinierten Alert-Dialog
	 * @param {string} message - Die anzuzeigende Nachricht
	 * @param {string} title - Optional: Titel des Dialogs (Standard: "Information")
	 */
	window.simpbookAlert = function(message, title) {
		title = title || 'Information';
		
		return new Promise(function(resolve) {
			// Erstelle Modal HTML mit positiver Klasse wenn Titel "Erfolg" enthält
			var isSuccess = title.toLowerCase().includes('erfolg') || title.toLowerCase().includes('success');
			var iconClass = isSuccess ? 'dashicons-yes-alt' : 'dashicons-info';
			var contentClass = isSuccess ? 'positive' : '';
			
			var modalHtml = '<div class="simpbook-dialog-modal active" id="simpbook-alert-modal">';
			modalHtml += '<div class="simpbook-dialog-content ' + contentClass + '">';
			modalHtml += '<div class="simpbook-dialog-header">' + title + '</div>';
			modalHtml += '<div class="simpbook-dialog-body">';
			modalHtml += '<span class="dashicons ' + iconClass + ' simpbook-dialog-icon"></span>';
			modalHtml += '<p>' + message + '</p>';
			modalHtml += '</div>';
			modalHtml += '<div class="simpbook-dialog-footer">';
		modalHtml += '<button class="simpbook-dialog-button simpbook-dialog-button-primary" id="simpbook-alert-ok">' + window.simpbookDialogConfig.buttonOk + '</button>';
			modalHtml += '</div>';
			modalHtml += '</div>';
			modalHtml += '</div>';
			
			// Füge Modal zum Body hinzu
			$('body').append(modalHtml);
			
			// Event Handler für OK-Button
			$('#simpbook-alert-ok').on('click', function() {
				$('#simpbook-alert-modal').remove();
				resolve();
			});
			
			// Schließe bei Klick außerhalb des Dialogs
			$('#simpbook-alert-modal').on('click', function(e) {
				if ($(e.target).hasClass('simpbook-dialog-modal')) {
					$('#simpbook-alert-modal').remove();
					resolve();
				}
			});
			
			// ESC-Taste zum Schließen
			$(document).on('keydown.simpbook-alert', function(e) {
				if (e.key === 'Escape' || e.keyCode === 27) {
					$('#simpbook-alert-modal').remove();
					$(document).off('keydown.simpbook-alert');
					resolve();
				}
			});
		});
	};
	
	/**
	 * Zeigt einen benutzerdefinierten Confirm-Dialog
	 * @param {string} message - Die anzuzeigende Nachricht
	 * @param {string} title - Optional: Titel des Dialogs (Standard: "Bestätigung")
	 * @returns {Promise<boolean>} - Promise, das true zurückgibt wenn bestätigt, false wenn abgebrochen
	 */
	window.simpbookConfirm = function(message, title) {
		title = title || 'Bestätigung';
		
		return new Promise(function(resolve) {
			// Erstelle Modal HTML mit negativer Klasse
			var modalHtml = '<div class="simpbook-dialog-modal active" id="simpbook-confirm-modal">';
			modalHtml += '<div class="simpbook-dialog-content negative">';
			modalHtml += '<div class="simpbook-dialog-header">' + title + '</div>';
			modalHtml += '<div class="simpbook-dialog-body">';
			modalHtml += '<span class="dashicons dashicons-warning simpbook-dialog-icon"></span>';
			modalHtml += '<p>' + message + '</p>';
			modalHtml += '</div>';
			modalHtml += '<div class="simpbook-dialog-footer">';
		modalHtml += '<button class="simpbook-dialog-button simpbook-dialog-button-secondary" id="simpbook-confirm-cancel">' + window.simpbookDialogConfig.buttonCancel + '</button>';
		modalHtml += '<button class="simpbook-dialog-button simpbook-dialog-button-danger" id="simpbook-confirm-ok">' + window.simpbookDialogConfig.buttonConfirm + '</button>';
			modalHtml += '</div>';
			modalHtml += '</div>';
			modalHtml += '</div>';
			
			// Füge Modal zum Body hinzu
			$('body').append(modalHtml);
			
			// Event Handler für Buttons
			$('#simpbook-confirm-ok').on('click', function() {
				$('#simpbook-confirm-modal').remove();
				$(document).off('keydown.simpbook-confirm');
				resolve(true);
			});
			
			$('#simpbook-confirm-cancel').on('click', function() {
				$('#simpbook-confirm-modal').remove();
				$(document).off('keydown.simpbook-confirm');
				resolve(false);
			});
			
			// Schließe bei Klick außerhalb des Dialogs (als Abbrechen behandeln)
			$('#simpbook-confirm-modal').on('click', function(e) {
				if ($(e.target).hasClass('simpbook-dialog-modal')) {
					$('#simpbook-confirm-modal').remove();
					$(document).off('keydown.simpbook-confirm');
					resolve(false);
				}
			});
			
			// ESC-Taste zum Abbrechen, Enter-Taste zum Bestätigen
			$(document).on('keydown.simpbook-confirm', function(e) {
				if (e.key === 'Escape' || e.keyCode === 27) {
					$('#simpbook-confirm-modal').remove();
					$(document).off('keydown.simpbook-confirm');
					resolve(false);
				} else if (e.key === 'Enter' || e.keyCode === 13) {
					$('#simpbook-confirm-modal').remove();
					$(document).off('keydown.simpbook-confirm');
					resolve(true);
				}
			});
		});
	};

	/**
	 * Zeigt einen benutzerdefinierten Confirm-Dialog für positive Aktionen (grün)
	 * @param {string} message - Die anzuzeigende Nachricht
	 * @param {string} title - Optional: Titel des Dialogs (Standard: "Bestätigung")
	 * @returns {Promise<boolean>} - Promise, das true zurückgibt wenn bestätigt, false wenn abgebrochen
	 */
	window.simpbookConfirmPositive = function(message, title) {
		title = title || 'Bestätigung';
		
		return new Promise(function(resolve) {
			// Erstelle Modal HTML mit positiver Klasse
			var modalHtml = '<div class="simpbook-dialog-modal active" id="simpbook-confirm-positive-modal">';
			modalHtml += '<div class="simpbook-dialog-content positive">';
			modalHtml += '<div class="simpbook-dialog-header">' + title + '</div>';
			modalHtml += '<div class="simpbook-dialog-body">';
			modalHtml += '<span class="dashicons dashicons-yes-alt simpbook-dialog-icon"></span>';
			modalHtml += '<p>' + message + '</p>';
			modalHtml += '</div>';
			modalHtml += '<div class="simpbook-dialog-footer">';
		modalHtml += '<button class="simpbook-dialog-button simpbook-dialog-button-secondary" id="simpbook-confirm-positive-cancel">' + window.simpbookDialogConfig.buttonCancel + '</button>';
		modalHtml += '<button class="simpbook-dialog-button simpbook-dialog-button-success" id="simpbook-confirm-positive-ok">' + window.simpbookDialogConfig.buttonConfirm + '</button>';
			modalHtml += '</div>';
			modalHtml += '</div>';
			modalHtml += '</div>';
			
			// Füge Modal zum Body hinzu
			$('body').append(modalHtml);
			
			// Event Handler für Buttons
			$('#simpbook-confirm-positive-ok').on('click', function() {
				$('#simpbook-confirm-positive-modal').remove();
				$(document).off('keydown.simpbook-confirm-positive');
				resolve(true);
			});
			
			$('#simpbook-confirm-positive-cancel').on('click', function() {
				$('#simpbook-confirm-positive-modal').remove();
				$(document).off('keydown.simpbook-confirm-positive');
				resolve(false);
			});
			
			// Schließe bei Klick außerhalb des Dialogs (als Abbrechen behandeln)
			$('#simpbook-confirm-positive-modal').on('click', function(e) {
				if ($(e.target).hasClass('simpbook-dialog-modal')) {
					$('#simpbook-confirm-positive-modal').remove();
					$(document).off('keydown.simpbook-confirm-positive');
					resolve(false);
				}
			});
			
			// ESC-Taste zum Abbrechen, Enter-Taste zum Bestätigen
			$(document).on('keydown.simpbook-confirm-positive', function(e) {
				if (e.key === 'Escape' || e.keyCode === 27) {
					$('#simpbook-confirm-positive-modal').remove();
					$(document).off('keydown.simpbook-confirm-positive');
					resolve(false);
				} else if (e.key === 'Enter' || e.keyCode === 13) {
					$('#simpbook-confirm-positive-modal').remove();
					$(document).off('keydown.simpbook-confirm-positive');
					resolve(true);
				}
			});
		});
	};

})(jQuery);


